/* --------------------------------------------------------------
  CreateStyleModal.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


/* globals -Modal */

'use strict';

import Modal from '../libs/Modal'; 
import EventsEmitter from '../libs/EventsEmitter'; 
import StyleApi from '../api/StyleApi'; 
import BoilerplateApi from '../api/BoilerplateApi';

/**
 * Create Style Layer Controller
 *
 * This class is responsible for the operation of the new style creation. Once the style is
 * created this controller will trigger the "StyleEdit.NewStyle" event so that other controllers
 * react accordingly.
 */
export default class CreateStyleModal {
	/**
	 * Class Constructor 
	 */
	constructor() {
		/**
		 * Modal Selector
		 *
		 * @type {jQuery}
		 */
		this.$modal = null;
	}
	
	/**
	 * Initialize Controller
	 *
	 * Fetch the available boilerplates and display them in a modal.
	 *
	 * @return {jQuery.Promise} Will be resolved once the method is ready.
	 */
	initialize() {
		const deferred = $.Deferred();
		
		BoilerplateApi.list().done((boilerplates) => {
			const data = {
				title_create_style_modal: StyleEdit.Language.translate('title_create_style_modal', 'style_edit'),
				placeholder_style_name: StyleEdit.Language.translate('placeholder_style_name', 'style_edit'),
				option_create: StyleEdit.Language.translate('option_create', 'style_edit'),
				option_cancel: StyleEdit.Language.translate('option_cancel', 'style_edit'),
				message_create_style_hint: StyleEdit.Language.translate('message_create_style_hint', 'style_edit'),
				available_boilerplates: boilerplates.config,
				template: StyleEdit.Config.get('template'),
				styles_base_path: StyleEdit.Config.get('theme') ? '../public/theme/styles/styleedit/' : 'templates/' + StyleEdit.Config.get('template') + '/'
			};
			
			StyleEdit.Language.addSection('template', boilerplates.lang);
			
			$.each(data.available_boilerplates, (index, boilerplate) => {
				boilerplate.title = StyleEdit.Language.translate(boilerplate.name, 'template');
			});
			
			this.$modal = Modal.show($('#create-style-modal-template'), data);
			this._bindEventHandlers();
			
			// Focus style-name input field
			this.$modal.find('input[name="style-name"]').focus();
			
			// Select the first boilerplate of the modal. 
			this.$modal.find('.boilerplates .preview-item:first').addClass('selected');
			
			deferred.resolve();
			
			EventsEmitter.triggerControllerInitialized(this.$modal, ['CreateStyleModal']);
			
			$.material.init();
		});
		
		return deferred.promise();
	}
	
	/**
	 * Destroy Controller
	 *
	 * Hide the create-style modal.
	 */
	destroy() {
		Modal.hide(this.$modal);
		EventsEmitter.triggerControllerDestroyed(this.$modal, ['CreateStyleModal']);
	}
	
	/**
	 * Event: Click create button.
	 *
	 * @param {jQuery.Event} event
	 *
	 * @private
	 */
	_onClickCreate(event) {
		const $element = $(event.currentTarget);
		const $newStyleName = this.$modal.find('input[name="style-name"]');
		
		$newStyleName.parent().removeClass('has-error');
		
		if ($newStyleName.val() === '') {
			$newStyleName.parent().addClass('has-error');
			$newStyleName.focus();
			return;
		}
		
		const boilerplateName = this.$modal.find('.boilerplates .preview-item.selected img').attr('data-name');
		const newStyleName = $newStyleName.val();
		
		StyleApi.create(boilerplateName, newStyleName).done((response) => {
			EventsEmitter.triggerStyleCreated($element, [response]);
			Modal.hide(this.$modal);
		});
	}
	
	/**
	 * Event: Click cancel button.
	 *
	 * Will hide the displayed modal.
	 *
	 * @private
	 */
	_onClickCancel() {
		Modal.hide(this.$modal);
	}
	
	/**
	 * Event: Click boilerplate preview image.
	 *
	 * Whenever the user clicks on the boilerplate preview image the "selected" class must
	 * be changed.
	 *
	 * @param {jQuery.Event} event
	 *
	 * @private
	 */
	_onClickBoilerplatePreview(event) {
		this.$modal.find('.boilerplates .preview-item.selected').removeClass('selected');
		$(event.currentTarget).parent().addClass('selected');
	}
	
	/**
	 * Bind Event Handlers
	 *
	 * @private
	 */
	_bindEventHandlers() {
		this.$modal
			.on('click', '.btn.create', event => this._onClickCreate(event))
			.on('click', '.btn.cancel', () => this._onClickCancel())
			.on('click', '.boilerplates .preview-item img', event => this._onClickBoilerplatePreview(event));
	}
}
